/**
 * Copyright 2021 Google Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * @fileoverview This is the application server that is loaded when you start
 * the server. It serves routes for both the Classroom add-on and the EdTech
 * company website.
 */

const createError = require('http-errors');
const express = require('express');
const { engine } = require('express-handlebars');
const cookieParser = require('cookie-parser');
const expressSession = require('express-session');
const path = require('path');
const logger = require('morgan');

const keys = require('../client_secret.json');

const websiteRouter = require('./routes/index');
const googleRouter = require('./routes/google');
const addonRouter = require('./routes/classroom-addon');

const app = express();

// WARNING : NOT FOR PRODUCTION
// ----------------------------
// This is a sample application for development purposes. You should follow
// best practices when securing your production application and in particular
// how you securely store and use OAuth tokens.
//
// Review these resources for additional security considerations
// + Google Identity developer website
//   https://developers.google.com/identity
// + OAuth 2.0 Security Best Current Practice
//   https://datatracker.ietf.org/doc/html/draft-ietf-oauth-security-topics
// + OAuth 2.0 Threat Model and Security Considerations
//   https://datatracker.ietf.org/doc/html/rfc6819
//
// Warning, express-session's default server-side session storage, MemoryStore,
// is purposely not designed for a production environment. It will leak memory
// under most conditions, does not scale past a single process, and is meant for
// debugging and developing.
app.use(expressSession({
  name: 'addon-session-CHANGEME', // Choose a unique name that is not guessable.
  resave: false,
  secret: keys.session.secret,
  saveUninitialized: false,
  cookie: {
    maxAge: 30 * 24 * 60 * 60 * 1000, // 30 days
    sameSite: 'none', // Needed because we are in an iframe.
    secure: true,
    httpOnly: true,
  }
}));

// Set up the view engine to use Handlebars.
app.engine('hbs', engine({
  extname: 'hbs',
  defaultLayout: 'base',
  defaultView: 'index',
  layoutsDir: `${__dirname}/views/layouts/`,
  partialsDir: `${__dirname}/views/partials/`
}));
app.set('view engine', 'hbs');
app.set(`views`, `${__dirname}/views`);

app.use(logger('dev'));
app.use(express.json());
app.use(express.urlencoded({ extended: false }));
app.use(cookieParser());
app.use(express.static(path.join(__dirname, 'public')));

app.use('/', websiteRouter);
app.use('/google', googleRouter);
app.use('/classroom-addon', addonRouter);

// Catch 404 and forward to error handler.
app.use(function (req, res, next) {
  next(createError(404));
});

// Error handler.
app.use(function (err, req, res, next) {
  // Set locals, only providing error in development.
  res.locals.message = err.message;
  res.locals.error = req.app.get('env') === 'development' ? err : {};

  // Render the error page.
  res.status(err.status || 500);
  res.render('error');
});

module.exports = app;
