/**
 * Copyright 2021 Google Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * @fileoverview This module serves all routes begining with "/user".
 */

const express = require('express');
const router = express.Router();
const checkAuthenticated = require('../middleware/checkAuthenticated');
const { getOAuthClient } = require('../clients/oauth2client');
const { google } = require('googleapis');

/**
 * /user
 *
 * Renders a page that lists the user's profile information. This route shows
 * making a Google UserInfo API call using the credentials received from
 * sign-in.
 */
router.get('/', checkAuthenticated, function (req, res, next) {
  const oauth2Client = getOAuthClient();
  oauth2Client.setCredentials(req.session.tokens);

  const oauth2 = google.oauth2("v2");
  async function service() {
    const { data } = await oauth2.userinfo.get({ auth: oauth2Client });
    res.render('user', {
      title: "User Info",
      data: JSON.stringify(data, null, 4)
    });
  }
  service().catch((err) => next(err));
});

module.exports = router;
