/**
 * Copyright 2021 Google Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * @fileoverview This file is a module with a set of helper functions to work
 * with Google APIs.
 */

const { google } = require('googleapis');
const keys = require('../../client_secret.json');

/**
 * getOAuthClient
 *
 * Created a new instance of the Google OAuth client using the secrets found in
 * client_secrets.json file downloaded from the Cloud Console for this Google
 * Cloud Project. client_secrets.json should NOT be commited to your source
 * control repository.
 *
 * @returns An new instance of the Google OAuth client.
 */
function getOAuthClient() {
  return new google.auth.OAuth2(
    keys.web.client_id,
    keys.web.client_secret,
    keys.web.redirect_uris[0],
  );
}

/**
 * getAuthUrl
 *
 * Builds a Google OAuth URL with the right scopes and parameters to sign-in and
 * authorize the user to use the add-on.
 *
 * @param {string} login_hint used to automatically sign the user in.
 * @param {string} hd used to limit the account choices in the sign-in pop-up.
 * @returns {string} Google URL to direct user to in order to sign-in and
 *   authenticate with the add-on.
 */
function getAuthUrl(login_hint, hd) {
  const oauth2Client = getOAuthClient();
  return oauth2Client.generateAuthUrl({
    // Enable offline access so that you can refresh an access token without
    // re-prompting the user for permission. Recommended for web server apps.
    // 'online' (default) or 'offline' (gets refresh_token)
    access_type: 'offline',
    // Needed to get the refreshToken on re-authorizations
    // prompt: 'consent',
    // Enable incremental authorization. Recommended as a best practice.
    include_granted_scopes: true,
    // Scopes needed by the add-on
    scope: [
      'https://www.googleapis.com/auth/userinfo.profile',
      'https://www.googleapis.com/auth/userinfo.email',
      "https://www.googleapis.com/auth/classroom.courses.readonly",
      "https://www.googleapis.com/auth/classroom.addons.student",
      "https://www.googleapis.com/auth/classroom.addons.teacher",
      // additional scopes as needed
    ],
    // If we have the login_hint, the user will automatically be selected.
    login_hint,
    // Without the login_hint, the hosted domain (hd) will reduce the list
    // of accounts to those just matching the domain.
    hd,
  });
}

exports.getOAuthClient = getOAuthClient;
exports.getAuthUrl = getAuthUrl;
